# AI dla biznesu — minimalizuj koszty dzięki głębokiemu Q-learningowi
# Tworzenie środowiska

# Importowanie bibliotek
import numpy as np

# TWORZENIE KLASY DLA ŚRODOWISKA

class Environment(object):
    
    # INICJALIZACJA WSZYSTKICH ZMIENNYCH DLA ŚRODOWISKA
    
    def __init__(self, optimal_temperature = (18.0, 24.0), initial_month = 0, initial_number_users = 10, initial_rate_data = 60):
        self.monthly_atmospheric_temperatures = [1.0, 5.0, 7.0, 10.0, 11.0, 20.0, 23.0, 24.0, 22.0, 10.0, 5.0, 1.0]
        self.initial_month = initial_month
        self.atmospheric_temperature = self.monthly_atmospheric_temperatures[initial_month]
        self.optimal_temperature = optimal_temperature
        self.min_temperature = -20
        self.max_temperature = 80
        self.min_number_users = 10
        self.max_number_users = 100
        self.max_update_users = 5
        self.min_rate_data = 20
        self.max_rate_data = 300
        self.max_update_data = 10
        self.initial_number_users = initial_number_users
        self.current_number_users = initial_number_users
        self.initial_rate_data = initial_rate_data
        self.current_rate_data = initial_rate_data
        self.intrinsic_temperature = self.atmospheric_temperature + 1.25 * self.current_number_users + 1.25 * self.current_rate_data
        self.temperature_ai = self.intrinsic_temperature
        self.temperature_noai = (self.optimal_temperature[0] + self.optimal_temperature[1]) / 2.0
        self.total_energy_ai = 0.0
        self.total_energy_noai = 0.0
        self.reward = 0.0
        self.game_over = 0
        self.train = 1

    # TWORZENIE METODY, KTÓRA AKTUALIZUJE ŚRODOWISKO PO WYKONANIU AKCJI PRZEZ AI
    
    def update_env(self, direction, energy_ai, month):
        
        # OBLICZANIE NAGRODY
        
        # Obliczanie ilości energii zużytej przez system chłodzenia serwera, gdy nie ma AI
        energy_noai = 0
        if (self.temperature_noai < self.optimal_temperature[0]):
            energy_noai = self.optimal_temperature[0] - self.temperature_noai
            self.temperature_noai = self.optimal_temperature[0]
        elif (self.temperature_noai > self.optimal_temperature[1]):
            energy_noai = self.temperature_noai - self.optimal_temperature[1]
            self.temperature_noai = self.optimal_temperature[1]
        # Obliczanie nagrody
        self.reward = energy_noai - energy_ai
        # Skalowanie nagrody
        self.reward = 1e-3 * self.reward
        
        # POBIERANIE NASTĘPNEGO STANU
        
        # Aktualizacja temperatury atmosferycznej
        self.atmospheric_temperature = self.monthly_atmospheric_temperatures[month]
        # Aktualizacja liczby użytkowników
        self.current_number_users += np.random.randint(-self.max_update_users, self.max_update_users)
        if (self.current_number_users > self.max_number_users):
            self.current_number_users = self.max_number_users
        elif (self.current_number_users < self.min_number_users):
            self.current_number_users = self.min_number_users
        # Aktualizacja częstotliwości danych
        self.current_rate_data += np.random.randint(-self.max_update_data, self.max_update_data)
        if (self.current_rate_data > self.max_rate_data):
            self.current_rate_data = self.max_rate_data
        elif (self.current_rate_data < self.min_rate_data):
            self.current_rate_data = self.min_rate_data
        # Obliczanie Delty Temperatury wewnętrzenej
        past_intrinsic_temperature = self.intrinsic_temperature
        self.intrinsic_temperature = self.atmospheric_temperature + 1.25 * self.current_number_users + 1.25 * self.current_rate_data
        delta_intrinsic_temperature = self.intrinsic_temperature - past_intrinsic_temperature
        # Obliczanie delty temperatury w wyniku działania AI
        if (direction == -1):
            delta_temperature_ai = -energy_ai
        elif (direction == 1):
            delta_temperature_ai = energy_ai
        # Aktualizacja nowej temperatury serwera, gdy jest AI
        self.temperature_ai += delta_intrinsic_temperature + delta_temperature_ai
        # Aktualizacja nowej temperatury serwera, gdy nie ma AI 
        self.temperature_noai += delta_intrinsic_temperature
        
        # ZAKOŃCZENIE ROZGRYWKI
        
        if (self.temperature_ai < self.min_temperature):
            if (self.train == 1):
                self.game_over = 1
            else:
                self.total_energy_ai += self.optimal_temperature[0] - self.temperature_ai
                self.temperature_ai = self.optimal_temperature[0]
        elif (self.temperature_ai > self.max_temperature):
            if (self.train == 1):
                self.game_over = 1
            else:
                self.total_energy_ai += self.temperature_ai - self.optimal_temperature[1]
                self.temperature_ai = self.optimal_temperature[1]
        
        # AKTUALIZACJA WYNIKÓW
        # Aktualizacja całkowitej energii zużywanej przez sztuczną inteligencję

        self.total_energy_ai += energy_ai
        # Aktualizacja całkowitej energii zużywanej przez system chłodzenia serwera, gdy nie ma AI
        self.total_energy_noai += energy_noai
        
        # SKALOWANIE NASTĘPNEGO STANU
        
        scaled_temperature_ai = (self.temperature_ai - self.min_temperature) / (self.max_temperature - self.min_temperature)
        scaled_number_users = (self.current_number_users - self.min_number_users) / (self.max_number_users - self.min_number_users)
        scaled_rate_data = (self.current_rate_data - self.min_rate_data) / (self.max_rate_data - self.min_rate_data)
        next_state = np.matrix([scaled_temperature_ai, scaled_number_users, scaled_rate_data])
        
        # ZWRÓCENIE NASTĘPNEGO STANU, NAGRODY I STATUSU ROZGRYWKI
        
        return next_state, self.reward, self.game_over

    # TWORZENIE METODY, KTÓRA RESETUJE ŚRODOWISKO
    
    def reset(self, new_month):
        self.atmospheric_temperature = self.monthly_atmospheric_temperatures[new_month]
        self.initial_month = new_month
        self.current_number_users = self.initial_number_users
        self.current_rate_data = self.initial_rate_data
        self.intrinsic_temperature = self.atmospheric_temperature + 1.25 * self.current_number_users + 1.25 * self.current_rate_data
        self.temperature_ai = self.intrinsic_temperature
        self.temperature_noai = (self.optimal_temperature[0] + self.optimal_temperature[1]) / 2.0
        self.total_energy_ai = 0.0
        self.total_energy_noai = 0.0
        self.reward = 0.0
        self.game_over = 0
        self.train = 1

    # TWORZENIE METODY, KTÓRA DAJE NAM W DOWOLNYM CZASIE BIEŻĄCY STAN, OSTATNIĄ NAGRODĘ ORAZ INFORMACJĘ, CZY ROZGRYWKA SIĘ ZAKOŃCZYŁA
    
    def observe(self):
        scaled_temperature_ai = (self.temperature_ai - self.min_temperature) / (self.max_temperature - self.min_temperature)
        scaled_number_users = (self.current_number_users - self.min_number_users) / (self.max_number_users - self.min_number_users)
        scaled_rate_data = (self.current_rate_data - self.min_rate_data) / (self.max_rate_data - self.min_rate_data)
        current_state = np.matrix([scaled_temperature_ai, scaled_number_users, scaled_rate_data])
        return current_state, self.reward, self.game_over
